<?php
/**
 * Premium Image Processor Class
 * 
 * Advanced image processing with:
 * - AWS Rekognition face landmark integration
 * - Precise passport photo positioning
 * - 4K quality output
 * - Print-ready sheets with crop marks
 * 
 * PHP 8.x compatible - all float to int conversions explicitly cast
 */

class ImageProcessor {
    
    /**
     * Load an image from file
     */
    public static function loadImage($path) {
        $info = getimagesize($path);
        if (!$info) return false;
        
        switch ($info[2]) {
            case IMAGETYPE_JPEG:
                return imagecreatefromjpeg($path);
            case IMAGETYPE_PNG:
                return imagecreatefrompng($path);
            case IMAGETYPE_WEBP:
                return imagecreatefromwebp($path);
            case IMAGETYPE_GIF:
                return imagecreatefromgif($path);
            default:
                return false;
        }
    }
    
    /**
     * Save an image with maximum quality
     */
    public static function saveImage($image, $path, $quality = null) {
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        $quality = $quality ?? (defined('OUTPUT_QUALITY_JPEG') ? OUTPUT_QUALITY_JPEG : 95);
        
        switch ($extension) {
            case 'jpg':
            case 'jpeg':
                imageinterlace($image, true);
                return imagejpeg($image, $path, (int)$quality);
            case 'png':
                imagesavealpha($image, true);
                $pngQuality = defined('OUTPUT_QUALITY_PNG') ? OUTPUT_QUALITY_PNG : 0;
                return imagepng($image, $path, (int)$pngQuality);
            case 'webp':
                return imagewebp($image, $path, (int)$quality);
            default:
                return imagejpeg($image, $path, (int)$quality);
        }
    }
    
    /**
     * Create passport photo using AWS Rekognition face data
     */
    public static function createPassportPhotoWithLandmarks($image, $spec, $faceData = null) {
        $srcWidth = imagesx($image);
        $srcHeight = imagesy($image);
        
        if ($faceData && isset($faceData['boundingBox'])) {
            $bbox = $faceData['boundingBox'];
            
            $faceLeft = (float)$bbox['Left'] * $srcWidth;
            $faceTop = (float)$bbox['Top'] * $srcHeight;
            $faceWidth = (float)$bbox['Width'] * $srcWidth;
            $faceHeight = (float)$bbox['Height'] * $srcHeight;
            
            $eyeCenter = null;
            if (isset($faceData['landmarks'])) {
                $leftEye = null;
                $rightEye = null;
                
                foreach ($faceData['landmarks'] as $landmark) {
                    if ($landmark['Type'] === 'eyeLeft') {
                        $leftEye = [
                            'x' => (float)$landmark['X'] * $srcWidth,
                            'y' => (float)$landmark['Y'] * $srcHeight,
                        ];
                    }
                    if ($landmark['Type'] === 'eyeRight') {
                        $rightEye = [
                            'x' => (float)$landmark['X'] * $srcWidth,
                            'y' => (float)$landmark['Y'] * $srcHeight,
                        ];
                    }
                }
                
                if ($leftEye && $rightEye) {
                    $eyeCenter = [
                        'x' => ($leftEye['x'] + $rightEye['x']) / 2,
                        'y' => ($leftEye['y'] + $rightEye['y']) / 2,
                    ];
                }
            }
            
            $headTop = max(0, $faceTop - ($faceHeight * 0.5));
            $headBottom = min($srcHeight, $faceTop + $faceHeight + ($faceHeight * 0.1));
            $headLeft = max(0, $faceLeft - ($faceWidth * 0.2));
            $headRight = min($srcWidth, $faceLeft + $faceWidth + ($faceWidth * 0.2));
            
            $headWidth = $headRight - $headLeft;
            $headHeight = $headBottom - $headTop;
            
            $targetHeadRatio = ($spec['head_height_min'] + $spec['head_height_max']) / 2;
            $targetHeadHeight = $spec['height'] * $targetHeadRatio;
            $scale = $headHeight > 0 ? $targetHeadHeight / $headHeight : 1;
            
            $cropWidth = $spec['width'] / $scale;
            $cropHeight = $spec['height'] / $scale;
            
            $faceCenterX = $faceLeft + ($faceWidth / 2);
            $cropX = $faceCenterX - ($cropWidth / 2);
            
            if ($eyeCenter) {
                $targetEyeY = $spec['height'] * (1 - $spec['eye_position']);
                $cropY = $eyeCenter['y'] - ($targetEyeY / $scale);
            } else {
                $faceCenterY = $faceTop + ($faceHeight / 2);
                $cropY = $faceCenterY - ($cropHeight * 0.45);
            }
            
            $cropX = max(0, min($srcWidth - $cropWidth, $cropX));
            $cropY = max(0, min($srcHeight - $cropHeight, $cropY));
            
        } else {
            $face = self::detectFaceBasic($image);
            
            if ($face) {
                $scale = min($srcWidth / $spec['width'], $srcHeight / $spec['height']) * 0.8;
                $cropWidth = $spec['width'] * $scale;
                $cropHeight = $spec['height'] * $scale;
                $cropX = ($face['x'] + $face['width'] / 2) - ($cropWidth / 2);
                $cropY = $face['y'] - ($cropHeight * 0.2);
            } else {
                $scale = min($srcWidth / $spec['width'], $srcHeight / $spec['height']);
                $cropWidth = $spec['width'] * $scale;
                $cropHeight = $spec['height'] * $scale;
                $cropX = ($srcWidth - $cropWidth) / 2;
                $cropY = ($srcHeight - $cropHeight) / 2;
            }
            
            $cropX = max(0, min($srcWidth - $cropWidth, $cropX));
            $cropY = max(0, min($srcHeight - $cropHeight, $cropY));
        }
        
        $passport = imagecreatetruecolor((int)$spec['width'], (int)$spec['height']);
        imageantialias($passport, true);
        
        $bgColor = imagecolorallocate(
            $passport,
            (int)$spec['bg_color'][0],
            (int)$spec['bg_color'][1],
            (int)$spec['bg_color'][2]
        );
        imagefill($passport, 0, 0, $bgColor);
        
        imagealphablending($passport, true);
        imagecopyresampled(
            $passport, $image,
            0, 0,
            (int)$cropX, (int)$cropY,
            (int)$spec['width'], (int)$spec['height'],
            (int)$cropWidth, (int)$cropHeight
        );
        
        return $passport;
    }
    
    /**
     * Basic face detection using skin color analysis (fallback)
     */
    public static function detectFaceBasic($image) {
        $width = imagesx($image);
        $height = imagesy($image);
        
        $skinPixels = [];
        $sampleStep = (int)max(1, (int)(min($width, $height) / 100));
        
        for ($y = 0; $y < $height; $y += $sampleStep) {
            for ($x = 0; $x < $width; $x += $sampleStep) {
                $rgb = imagecolorat($image, (int)$x, (int)$y);
                $r = ($rgb >> 16) & 0xFF;
                $g = ($rgb >> 8) & 0xFF;
                $b = $rgb & 0xFF;
                
                if (self::isSkinColor($r, $g, $b)) {
                    $skinPixels[] = ['x' => (int)$x, 'y' => (int)$y];
                }
            }
        }
        
        if (count($skinPixels) < 10) return null;
        
        $minX = $width;
        $maxX = 0;
        $minY = $height;
        $maxY = 0;
        
        foreach ($skinPixels as $pixel) {
            $minX = min($minX, $pixel['x']);
            $maxX = max($maxX, $pixel['x']);
            $minY = min($minY, $pixel['y']);
            $maxY = max($maxY, $pixel['y']);
        }
        
        return [
            'x' => (int)max(0, $minX - 20),
            'y' => (int)max(0, $minY - 20),
            'width' => (int)min($width - $minX, $maxX - $minX + 40),
            'height' => (int)min($height - $minY, ($maxY - $minY) * 0.6 + 40),
        ];
    }
    
    /**
     * Check if RGB is skin color
     */
    private static function isSkinColor($r, $g, $b) {
        if ($r < 60 || $g < 40 || $b < 20) return false;
        if (max($r, $g, $b) - min($r, $g, $b) < 15) return false;
        if (abs($r - $g) < 15) return false;
        if ($r <= $g || $r <= $b) return false;
        
        if ($r > 95 && $g > 40 && $b > 20 && 
            max($r, $g, $b) - min($r, $g, $b) > 15 && 
            abs($r - $g) > 15 && $r > $g && $r > $b) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Create print-ready photo sheet with crop marks
     */
    public static function createPhotoSheetPremium($photo, $cols = 2, $rows = 3, $options = []) {
        $defaults = [
            'gap' => 20,
            'margin' => 40,
            'cropMarks' => true,
            'cropMarkLength' => 15,
            'cropMarkOffset' => 5,
            'showDimensions' => true,
            'dpi' => 300,
        ];
        
        $options = array_merge($defaults, $options);
        
        $photoWidth = imagesx($photo);
        $photoHeight = imagesy($photo);
        
        $contentWidth = ($photoWidth * $cols) + ($options['gap'] * ($cols - 1));
        $contentHeight = ($photoHeight * $rows) + ($options['gap'] * ($rows - 1));
        
        $sheetWidth = (int)($contentWidth + ($options['margin'] * 2));
        $sheetHeight = (int)($contentHeight + ($options['margin'] * 2));
        
        $sheet = imagecreatetruecolor($sheetWidth, $sheetHeight);
        imageantialias($sheet, true);
        
        $white = imagecolorallocate($sheet, 255, 255, 255);
        $black = imagecolorallocate($sheet, 0, 0, 0);
        $gray = imagecolorallocate($sheet, 150, 150, 150);
        imagefill($sheet, 0, 0, $white);
        
        for ($row = 0; $row < $rows; $row++) {
            for ($col = 0; $col < $cols; $col++) {
                $x = (int)($options['margin'] + ($col * ($photoWidth + $options['gap'])));
                $y = (int)($options['margin'] + ($row * ($photoHeight + $options['gap'])));
                
                imagecopy($sheet, $photo, $x, $y, 0, 0, $photoWidth, $photoHeight);
                
                if ($options['cropMarks']) {
                    self::drawCropMarks($sheet, $x, $y, $photoWidth, $photoHeight, $gray, $options);
                }
            }
        }
        
        if ($options['showDimensions']) {
            $fontSize = 3;
            $widthMM = round($photoWidth / $options['dpi'] * 25.4, 1);
            $heightMM = round($photoHeight / $options['dpi'] * 25.4, 1);
            $label = $widthMM . 'mm x ' . $heightMM . 'mm';
            imagestring($sheet, $fontSize, 10, 10, $label, $gray);
        }
        
        return $sheet;
    }
    
    /**
     * Draw crop marks around a photo
     */
    private static function drawCropMarks($image, $x, $y, $width, $height, $color, $options) {
        $len = (int)$options['cropMarkLength'];
        $off = (int)$options['cropMarkOffset'];
        $x = (int)$x;
        $y = (int)$y;
        $width = (int)$width;
        $height = (int)$height;
        
        // Top-left
        imageline($image, $x - $off - $len, $y, $x - $off, $y, $color);
        imageline($image, $x, $y - $off - $len, $x, $y - $off, $color);
        
        // Top-right
        imageline($image, $x + $width + $off, $y, $x + $width + $off + $len, $y, $color);
        imageline($image, $x + $width, $y - $off - $len, $x + $width, $y - $off, $color);
        
        // Bottom-left
        imageline($image, $x - $off - $len, $y + $height, $x - $off, $y + $height, $color);
        imageline($image, $x, $y + $height + $off, $x, $y + $height + $off + $len, $color);
        
        // Bottom-right
        imageline($image, $x + $width + $off, $y + $height, $x + $width + $off + $len, $y + $height, $color);
        imageline($image, $x + $width, $y + $height + $off, $x + $width, $y + $height + $off + $len, $color);
    }
    
    /**
     * Create a 4x6 inch print sheet (standard photo print size)
     */
    public static function create4x6Sheet($photo, $spec) {
        $sheetWidth = 1800;  // 6 inches at 300 DPI
        $sheetHeight = 1200; // 4 inches at 300 DPI
        
        $photoWidth = (int)$spec['width'];
        $photoHeight = (int)$spec['height'];
        
        $cols = (int)floor(($sheetWidth - 60) / ($photoWidth + 20));
        $rows = (int)floor(($sheetHeight - 60) / ($photoHeight + 20));
        
        // Ensure at least 1 row and column
        $cols = max(1, $cols);
        $rows = max(1, $rows);
        
        $sheet = imagecreatetruecolor($sheetWidth, $sheetHeight);
        imageantialias($sheet, true);
        
        $white = imagecolorallocate($sheet, 255, 255, 255);
        imagefill($sheet, 0, 0, $white);
        
        $totalPhotosWidth = ($cols * $photoWidth) + (($cols - 1) * 20);
        $totalPhotosHeight = ($rows * $photoHeight) + (($rows - 1) * 20);
        $startX = (int)(($sheetWidth - $totalPhotosWidth) / 2);
        $startY = (int)(($sheetHeight - $totalPhotosHeight) / 2);
        
        for ($row = 0; $row < $rows; $row++) {
            for ($col = 0; $col < $cols; $col++) {
                $x = (int)($startX + ($col * ($photoWidth + 20)));
                $y = (int)($startY + ($row * ($photoHeight + 20)));
                imagecopy($sheet, $photo, $x, $y, 0, 0, $photoWidth, $photoHeight);
            }
        }
        
        return $sheet;
    }
    
    /**
     * Advanced image enhancement
     */
    public static function enhancePremium($image, $options = []) {
        $defaults = [
            'brightness' => 0,
            'contrast' => 0,
            'saturation' => 0,
            'sharpen' => true,
            'denoise' => false,
            'autoLevels' => true,
            'whiteBalance' => true,
        ];
        
        $options = array_merge($defaults, $options);
        
        if ($options['autoLevels']) {
            $image = self::autoLevels($image);
        }
        
        if ($options['whiteBalance']) {
            $image = self::autoWhiteBalance($image);
        }
        
        if ($options['brightness'] !== 0) {
            imagefilter($image, IMG_FILTER_BRIGHTNESS, (int)$options['brightness']);
        }
        
        if ($options['contrast'] !== 0) {
            imagefilter($image, IMG_FILTER_CONTRAST, (int)(-$options['contrast']));
        }
        
        if ($options['sharpen']) {
            $image = self::unsharpMask($image, 80, 0.5, 3);
        }
        
        if ($options['denoise']) {
            imagefilter($image, IMG_FILTER_SMOOTH, 3);
        }
        
        return $image;
    }
    
    /**
     * Auto levels adjustment
     */
    public static function autoLevels($image) {
        $width = imagesx($image);
        $height = imagesy($image);
        
        $histogram = ['r' => array_fill(0, 256, 0), 'g' => array_fill(0, 256, 0), 'b' => array_fill(0, 256, 0)];
        
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $rgb = imagecolorat($image, $x, $y);
                $histogram['r'][($rgb >> 16) & 0xFF]++;
                $histogram['g'][($rgb >> 8) & 0xFF]++;
                $histogram['b'][$rgb & 0xFF]++;
            }
        }
        
        $totalPixels = $width * $height;
        $clipPercent = 0.01;
        $clipPixels = $totalPixels * $clipPercent;
        
        $ranges = [];
        foreach (['r', 'g', 'b'] as $channel) {
            $count = 0;
            $min = 0;
            for ($i = 0; $i < 256; $i++) {
                $count += $histogram[$channel][$i];
                if ($count > $clipPixels) {
                    $min = $i;
                    break;
                }
            }
            
            $count = 0;
            $max = 255;
            for ($i = 255; $i >= 0; $i--) {
                $count += $histogram[$channel][$i];
                if ($count > $clipPixels) {
                    $max = $i;
                    break;
                }
            }
            
            $ranges[$channel] = ['min' => $min, 'max' => $max];
        }
        
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $rgb = imagecolorat($image, $x, $y);
                $a = ($rgb >> 24) & 0xFF;
                $r = self::stretchValue(($rgb >> 16) & 0xFF, $ranges['r']['min'], $ranges['r']['max']);
                $g = self::stretchValue(($rgb >> 8) & 0xFF, $ranges['g']['min'], $ranges['g']['max']);
                $b = self::stretchValue($rgb & 0xFF, $ranges['b']['min'], $ranges['b']['max']);
                
                $newColor = imagecolorallocatealpha($image, $r, $g, $b, $a);
                imagesetpixel($image, $x, $y, $newColor);
            }
        }
        
        return $image;
    }
    
    private static function stretchValue($value, $min, $max) {
        if ($max <= $min) return (int)$value;
        return (int)max(0, min(255, (int)((($value - $min) * 255) / ($max - $min))));
    }
    
    /**
     * Auto white balance
     */
    public static function autoWhiteBalance($image) {
        $width = imagesx($image);
        $height = imagesy($image);
        
        $rSum = $gSum = $bSum = 0;
        $count = 0;
        
        for ($y = 0; $y < $height; $y += 10) {
            for ($x = 0; $x < $width; $x += 10) {
                $rgb = imagecolorat($image, $x, $y);
                $rSum += ($rgb >> 16) & 0xFF;
                $gSum += ($rgb >> 8) & 0xFF;
                $bSum += $rgb & 0xFF;
                $count++;
            }
        }
        
        if ($count === 0) return $image;
        
        $rAvg = $rSum / $count;
        $gAvg = $gSum / $count;
        $bAvg = $bSum / $count;
        
        $avg = ($rAvg + $gAvg + $bAvg) / 3;
        
        $rScale = $avg / max(1, $rAvg);
        $gScale = $avg / max(1, $gAvg);
        $bScale = $avg / max(1, $bAvg);
        
        $rScale = max(0.8, min(1.2, $rScale));
        $gScale = max(0.8, min(1.2, $gScale));
        $bScale = max(0.8, min(1.2, $bScale));
        
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $rgb = imagecolorat($image, $x, $y);
                $a = ($rgb >> 24) & 0xFF;
                $r = (int)min(255, (int)((($rgb >> 16) & 0xFF) * $rScale));
                $g = (int)min(255, (int)((($rgb >> 8) & 0xFF) * $gScale));
                $b = (int)min(255, (int)(($rgb & 0xFF) * $bScale));
                
                $newColor = imagecolorallocatealpha($image, $r, $g, $b, $a);
                imagesetpixel($image, $x, $y, $newColor);
            }
        }
        
        return $image;
    }
    
    /**
     * Unsharp mask for sharpening
     */
    public static function unsharpMask($image, $amount = 80, $radius = 0.5, $threshold = 3) {
        $sharpenMatrix = [
            [-1, -1, -1],
            [-1, 16, -1],
            [-1, -1, -1],
        ];
        $divisor = 8;
        
        imageconvolution($image, $sharpenMatrix, $divisor, 0);
        
        return $image;
    }
    
    /**
     * Get image dimensions
     */
    public static function getDimensions($path) {
        $info = getimagesize($path);
        if (!$info) return false;
        return [(int)$info[0], (int)$info[1]];
    }
    
    /**
     * Resize image maintaining aspect ratio
     */
    public static function resize($image, $maxWidth, $maxHeight) {
        $srcWidth = imagesx($image);
        $srcHeight = imagesy($image);
        
        $ratio = min($maxWidth / $srcWidth, $maxHeight / $srcHeight);
        
        $newWidth = (int)($srcWidth * $ratio);
        $newHeight = (int)($srcHeight * $ratio);
        
        $newImage = imagecreatetruecolor($newWidth, $newHeight);
        imagealphablending($newImage, false);
        imagesavealpha($newImage, true);
        $transparent = imagecolorallocatealpha($newImage, 0, 0, 0, 127);
        imagefill($newImage, 0, 0, $transparent);
        
        imagecopyresampled($newImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $srcWidth, $srcHeight);
        
        return $newImage;
    }
}
