<?php
/**
 * PhotoAid - Process and return preview
 * Pure GD implementation with proper alpha compositing
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
header('Content-Type: application/json');

try {
    require_once __DIR__ . '/../includes/config.php';
    require_once __DIR__ . '/../includes/AIServices.php';
    require_once __DIR__ . '/../includes/ImageProcessor.php';
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('POST required');
    }
    
    if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Image upload failed');
    }
    
    // Get spec
    $documentType = $_POST['document_type'] ?? 'uk_passport';
    global $PASSPORT_SPECS;
    
    if (!isset($PASSPORT_SPECS[$documentType])) {
        $documentType = 'uk_passport';
    }
    $spec = $PASSPORT_SPECS[$documentType];
    
    $tempDir = TEMP_DIR;
    if (!file_exists($tempDir)) {
        mkdir($tempDir, 0755, true);
    }
    
    // Create processed directory for previews
    $previewDir = __DIR__ . '/previews/';
    if (!file_exists($previewDir)) {
        mkdir($previewDir, 0755, true);
    }
    
    $sessionId = uniqid('sess_', true);
    
    // Save uploaded file
    $extension = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION)) ?: 'jpg';
    $originalPath = $tempDir . $sessionId . '_original.' . $extension;
    
    if (!move_uploaded_file($_FILES['image']['tmp_name'], $originalPath)) {
        throw new Exception('Failed to save uploaded file');
    }
    
    // Remove background - get transparent PNG
    $noBgPath = $tempDir . $sessionId . '_nobg.png';
    $bgResult = AIServices::removeBackgroundPro($originalPath, $noBgPath, [
        'size' => 'full',
        'format' => 'png',
    ]);
    
    if (!$bgResult['success']) {
        @unlink($originalPath);
        throw new Exception('Background removal failed: ' . ($bgResult['error'] ?? 'API error'));
    }
    
    // Composite PNG onto pure white background using GD
    $compositedPath = $tempDir . $sessionId . '_white.jpg';
    flattenPngToWhite($noBgPath, $compositedPath);
    
    // Face detection (optional)
    $faceData = null;
    if (defined('AWS_ACCESS_KEY') && AWS_ACCESS_KEY !== 'YOUR_AWS_ACCESS_KEY') {
        $faceResult = AIServices::detectFaceAWS($originalPath);
        if ($faceResult['success']) {
            $faceData = $faceResult['face'];
        }
    }
    
    // Create passport photo from white-background image
    $image = ImageProcessor::loadImage($compositedPath);
    if (!$image) {
        throw new Exception('Failed to load composited image');
    }
    
    $passport = ImageProcessor::createPassportPhotoWithLandmarks($image, $spec, $faceData);
    $passport = ImageProcessor::enhancePremium($passport, [
        'sharpen' => true,
        'autoLevels' => true,
        'whiteBalance' => true,
    ]);
    
    // Save final passport photo (for later delivery)
    $passportPath = $tempDir . $sessionId . '_passport.jpg';
    ImageProcessor::saveImage($passport, $passportPath, 98);
    
    // Create photo sheet (for later delivery)
    $sheet = ImageProcessor::createPhotoSheetPremium($passport, 2, 3, [
        'cropMarks' => true,
        'showDimensions' => true,
    ]);
    $sheetPath = $tempDir . $sessionId . '_sheet.jpg';
    ImageProcessor::saveImage($sheet, $sheetPath, 98);
    
    // Create watermarked preview
    $previewPath = $previewDir . $sessionId . '_preview.jpg';
    $preview = addWatermark($passport);
    ImageProcessor::saveImage($preview, $previewPath, 85);
    
    // Run AI compliance check on the final passport photo
    $complianceResult = AIServices::checkUKPassportCompliance($passportPath);
    $compliance = $complianceResult['compliance'] ?? null;
    
    // Cleanup GD resources
    imagedestroy($image);
    imagedestroy($passport);
    imagedestroy($sheet);
    imagedestroy($preview);
    
    // Store session info
    $sessionData = [
        'session_id' => $sessionId,
        'document_type' => $documentType,
        'spec_name' => $spec['name'],
        'passport_path' => $passportPath,
        'sheet_path' => $sheetPath,
        'compliance' => $compliance,
        'created' => time(),
    ];
    file_put_contents($tempDir . $sessionId . '_session.json', json_encode($sessionData));
    
    // Return preview URL with compliance data
    echo json_encode([
        'success' => true,
        'session_id' => $sessionId,
        'preview_url' => 'previews/' . $sessionId . '_preview.jpg',
        'document_type' => $spec['name'],
        'compliance' => $compliance,
    ]);
    
} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

/**
 * Flatten transparent PNG onto pure white background
 * Pixel-by-pixel alpha compositing for correct results
 */
function flattenPngToWhite($pngPath, $outputPath) {
    // Load PNG with full alpha support
    $png = @imagecreatefrompng($pngPath);
    if (!$png) {
        throw new Exception('Failed to load PNG file');
    }
    
    $width = imagesx($png);
    $height = imagesy($png);
    
    // Create output image - true color, no alpha
    $output = imagecreatetruecolor($width, $height);
    
    // Fill with pure white first
    $white = imagecolorallocate($output, 255, 255, 255);
    imagefilledrectangle($output, 0, 0, $width - 1, $height - 1, $white);
    
    // Process each pixel - proper alpha blending onto white
    for ($y = 0; $y < $height; $y++) {
        for ($x = 0; $x < $width; $x++) {
            $rgba = imagecolorat($png, $x, $y);
            
            // Extract RGBA components
            // In GD: alpha 0 = fully opaque, 127 = fully transparent
            $srcR = ($rgba >> 16) & 0xFF;
            $srcG = ($rgba >> 8) & 0xFF;
            $srcB = $rgba & 0xFF;
            $srcA = ($rgba >> 24) & 0x7F;
            
            if ($srcA === 127) {
                // Fully transparent - leave as white (already filled)
                continue;
            }
            
            if ($srcA === 0) {
                // Fully opaque - use source color directly
                $color = imagecolorallocate($output, $srcR, $srcG, $srcB);
                imagesetpixel($output, $x, $y, $color);
            } else {
                // Semi-transparent - blend with white (255, 255, 255)
                // Convert GD alpha (0-127) to normalized opacity (1.0-0.0)
                $opacity = 1.0 - ($srcA / 127.0);
                
                // Alpha composite: result = src * opacity + dst * (1 - opacity)
                // dst is white (255, 255, 255)
                $dstR = 255;
                $dstG = 255;
                $dstB = 255;
                
                $finalR = (int)round($srcR * $opacity + $dstR * (1 - $opacity));
                $finalG = (int)round($srcG * $opacity + $dstG * (1 - $opacity));
                $finalB = (int)round($srcB * $opacity + $dstB * (1 - $opacity));
                
                // Clamp to valid range
                $finalR = max(0, min(255, $finalR));
                $finalG = max(0, min(255, $finalG));
                $finalB = max(0, min(255, $finalB));
                
                $color = imagecolorallocate($output, $finalR, $finalG, $finalB);
                imagesetpixel($output, $x, $y, $color);
            }
        }
    }
    
    imagedestroy($png);
    
    // Save as high-quality JPEG (no alpha channel possible)
    $result = imagejpeg($output, $outputPath, 98);
    imagedestroy($output);
    
    if (!$result) {
        throw new Exception('Failed to save composited image');
    }
}

/**
 * Add watermark to image
 */
function addWatermark($image) {
    $width = imagesx($image);
    $height = imagesy($image);
    
    // Create copy
    $watermarked = imagecreatetruecolor($width, $height);
    imagecopy($watermarked, $image, 0, 0, 0, 0, $width, $height);
    
    // Enable alpha blending for transparent overlays
    imagealphablending($watermarked, true);
    
    // Add diagonal watermark text
    $textColor = imagecolorallocatealpha($watermarked, 249, 115, 22, 80);
    
    $text = 'PREVIEW';
    $fontSize = 5;
    
    for ($y = -$height; $y < $height * 2; $y += 80) {
        for ($x = -$width; $x < $width * 2; $x += 150) {
            $rotX = (int)($x + $y * 0.5);
            $rotY = (int)($y - $x * 0.3);
            if ($rotX > 0 && $rotX < $width && $rotY > 0 && $rotY < $height) {
                imagestring($watermarked, $fontSize, $rotX, $rotY, $text, $textColor);
            }
        }
    }
    
    // Add semi-transparent overlay bar
    $overlay = imagecolorallocatealpha($watermarked, 0, 0, 0, 100);
    imagefilledrectangle($watermarked, 0, (int)($height * 0.45), $width, (int)($height * 0.55), $overlay);
    
    // Add center watermark text
    $centerText = 'PREVIEW - WATERMARKED';
    $textWidth = imagefontwidth($fontSize) * strlen($centerText);
    $textX = (int)(($width - $textWidth) / 2);
    $textY = (int)($height / 2 - 8);
    
    $whiteText = imagecolorallocate($watermarked, 255, 255, 255);
    imagestring($watermarked, $fontSize, $textX, $textY, $centerText, $whiteText);
    
    return $watermarked;
}
