<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Processing - PhotoAid</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        :root {
            --background: hsl(0 0% 100%);
            --foreground: hsl(220 20% 10%);
            --primary: hsl(24 95% 53%);
            --muted: hsl(220 14% 96%);
            --muted-foreground: hsl(220 10% 45%);
            --border: hsl(220 14% 90%);
            --gradient-primary: linear-gradient(135deg, hsl(38 92% 50%), hsl(24 95% 53%), hsl(12 90% 60%));
        }
        
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Inter', system-ui, sans-serif;
            background: var(--background);
            color: var(--foreground);
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 24px;
        }
        
        .gradient-text {
            background: var(--gradient-primary);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .card {
            background: white;
            border: 1px solid var(--border);
            border-radius: 24px;
            padding: 48px;
            max-width: 420px;
            width: 100%;
            text-align: center;
            box-shadow: 0 20px 40px -12px hsla(220, 20%, 10%, 0.1);
        }
        
        .spinner {
            width: 64px;
            height: 64px;
            border: 4px solid var(--border);
            border-top-color: var(--primary);
            border-radius: 50%;
            margin: 0 auto 24px;
            animation: spin 1s linear infinite;
        }
        @keyframes spin { to { transform: rotate(360deg); } }
        
        h1 {
            font-size: 24px;
            font-weight: 700;
            margin-bottom: 8px;
        }
        
        .subtitle {
            color: var(--muted-foreground);
            font-size: 16px;
            margin-bottom: 32px;
        }
        
        .steps-list {
            text-align: left;
        }
        
        .step-item {
            display: flex;
            align-items: center;
            gap: 16px;
            padding: 14px 0;
            border-bottom: 1px solid var(--border);
        }
        .step-item:last-child { border-bottom: none; }
        
        .step-indicator {
            width: 28px;
            height: 28px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            font-weight: 600;
            flex-shrink: 0;
        }
        
        .step-item.pending .step-indicator {
            background: var(--muted);
            color: var(--muted-foreground);
        }
        .step-item.active .step-indicator {
            background: var(--primary);
            color: white;
            animation: pulse 1s infinite;
        }
        .step-item.completed .step-indicator {
            background: hsl(142, 76%, 36%);
            color: white;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.1); }
        }
        
        .step-text {
            font-size: 14px;
            color: var(--muted-foreground);
        }
        .step-item.active .step-text,
        .step-item.completed .step-text {
            color: var(--foreground);
        }
        
        .logo {
            margin-top: 48px;
            font-size: 18px;
            font-weight: 700;
            color: var(--muted-foreground);
        }
        
        .error-message {
            display: none;
            background: hsla(0, 84%, 60%, 0.1);
            color: hsl(0, 84%, 45%);
            padding: 16px;
            border-radius: 12px;
            margin-top: 24px;
            font-size: 14px;
        }
        .error-message.show { display: block; }
        
        .retry-btn {
            display: none;
            margin-top: 16px;
            padding: 12px 24px;
            background: var(--primary);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
        }
        .retry-btn.show { display: inline-block; }
    </style>
</head>
<body>
    <div class="card">
        <div class="spinner" id="spinner"></div>
        <h1>Processing your photo</h1>
        <p class="subtitle">This only takes a few seconds</p>
        
        <div class="steps-list">
            <div class="step-item active" data-step="1">
                <div class="step-indicator">1</div>
                <span class="step-text">Analyzing face position...</span>
            </div>
            <div class="step-item pending" data-step="2">
                <div class="step-indicator">2</div>
                <span class="step-text">Removing background...</span>
            </div>
            <div class="step-item pending" data-step="3">
                <div class="step-indicator">3</div>
                <span class="step-text">Checking compliance...</span>
            </div>
            <div class="step-item pending" data-step="4">
                <div class="step-indicator">4</div>
                <span class="step-text">Adjusting lighting...</span>
            </div>
            <div class="step-item pending" data-step="5">
                <div class="step-indicator">5</div>
                <span class="step-text">Creating passport photo...</span>
            </div>
            <div class="step-item pending" data-step="6">
                <div class="step-indicator">6</div>
                <span class="step-text">Finalizing...</span>
            </div>
        </div>
        
        <div class="error-message" id="errorMessage"></div>
        <button class="retry-btn" id="retryBtn" onclick="window.location.href='index.php'">Try Again</button>
    </div>
    <div class="logo">photo<span class="gradient-text">Aid</span></div>

    <script>
        // Get file data from sessionStorage
        const fileData = sessionStorage.getItem('uploadedFile');
        const documentType = sessionStorage.getItem('documentType') || 'uk_passport';
        
        if (!fileData) {
            window.location.href = 'index.php';
        }
        
        // Convert base64 back to blob
        function base64ToBlob(base64, type) {
            const binary = atob(base64.split(',')[1]);
            const array = [];
            for (let i = 0; i < binary.length; i++) {
                array.push(binary.charCodeAt(i));
            }
            return new Blob([new Uint8Array(array)], { type });
        }
        
        function setStepActive(num) {
            const step = document.querySelector(`.step-item[data-step="${num}"]`);
            if (step) {
                step.classList.remove('pending');
                step.classList.add('active');
            }
        }
        
        function setStepCompleted(num) {
            const step = document.querySelector(`.step-item[data-step="${num}"]`);
            if (step) {
                step.classList.remove('pending', 'active');
                step.classList.add('completed');
                step.querySelector('.step-indicator').innerHTML = '✓';
            }
        }
        
        function showError(message) {
            document.getElementById('spinner').style.display = 'none';
            document.getElementById('errorMessage').textContent = message;
            document.getElementById('errorMessage').classList.add('show');
            document.getElementById('retryBtn').classList.add('show');
        }
        
        async function processPhoto() {
            const blob = base64ToBlob(fileData, 'image/jpeg');
            const formData = new FormData();
            formData.append('image', blob, 'photo.jpg');
            formData.append('document_type', documentType);
            
            // Animate steps
            let currentStep = 1;
            const stepInterval = setInterval(() => {
                if (currentStep <= 6) {
                    setStepCompleted(currentStep);
                    currentStep++;
                    if (currentStep <= 6) setStepActive(currentStep);
                }
            }, 700);
            
            try {
                const response = await fetch('process_preview.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                clearInterval(stepInterval);
                
                // Complete all steps
                for (let i = 1; i <= 6; i++) setStepCompleted(i);
                
                setTimeout(() => {
                    if (result.success) {
                        // Clear storage and redirect to results
                        sessionStorage.removeItem('uploadedFile');
                        sessionStorage.removeItem('documentType');
                        window.location.href = `results.php?session=${result.session_id}&preview=${encodeURIComponent(result.preview_url)}`;
                    } else {
                        showError(result.error || 'Processing failed. Please try again.');
                    }
                }, 500);
                
            } catch (error) {
                clearInterval(stepInterval);
                showError('Connection error. Please check your internet and try again.');
            }
        }
        
        // Start processing
        processPhoto();
    </script>
</body>
</html>
