<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Your Photo is Ready - PhotoAid</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        :root {
            --background: hsl(0 0% 100%);
            --foreground: hsl(220 20% 10%);
            --card: hsl(0 0% 100%);
            --primary: hsl(24 95% 53%);
            --primary-foreground: hsl(0 0% 100%);
            --secondary: hsl(220 14% 96%);
            --muted: hsl(220 14% 96%);
            --muted-foreground: hsl(220 10% 45%);
            --border: hsl(220 14% 90%);
            --radius: 1rem;
            --amber: hsl(38 92% 50%);
            --coral: hsl(12 90% 60%);
            --navy: hsl(220 40% 13%);
            --gradient-primary: linear-gradient(135deg, hsl(38 92% 50%), hsl(24 95% 53%), hsl(12 90% 60%));
            --shadow-sm: 0 2px 8px -2px hsla(220, 20%, 10%, 0.08);
            --shadow-md: 0 8px 24px -8px hsla(220, 20%, 10%, 0.12);
            --shadow-lg: 0 20px 40px -12px hsla(220, 20%, 10%, 0.15);
            --shadow-glow: 0 0 40px -10px hsla(24, 95%, 53%, 0.4);
        }
        
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Inter', system-ui, sans-serif;
            background: var(--background);
            color: var(--foreground);
            -webkit-font-smoothing: antialiased;
            min-height: 100vh;
        }
        
        .gradient-text {
            background: var(--gradient-primary);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        /* Header */
        .header {
            position: sticky;
            top: 0;
            background: hsla(0, 0%, 100%, 0.8);
            backdrop-filter: blur(20px);
            border-bottom: 1px solid var(--border);
            z-index: 50;
        }
        .header-inner {
            max-width: 1280px;
            margin: 0 auto;
            padding: 0 16px;
            height: 64px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        .back-btn {
            display: flex;
            align-items: center;
            gap: 8px;
            color: var(--muted-foreground);
            text-decoration: none;
            font-size: 14px;
            transition: color 0.2s;
        }
        .back-btn:hover { color: var(--foreground); }
        .logo {
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
            color: inherit;
            font-size: 20px;
            font-weight: 700;
        }
        
        /* Main Content */
        .main {
            max-width: 1000px;
            margin: 0 auto;
            padding: 48px 16px 96px;
        }
        
        /* Success Header */
        .success-header {
            text-align: center;
            margin-bottom: 48px;
        }
        .success-icon {
            width: 64px;
            height: 64px;
            border-radius: 50%;
            background: hsla(142, 76%, 36%, 0.15);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 16px;
        }
        .success-icon svg { width: 32px; height: 32px; color: hsl(142, 76%, 36%); }
        .success-header h1 {
            font-size: 32px;
            font-weight: 700;
            margin-bottom: 8px;
        }
        .success-header p {
            color: var(--muted-foreground);
            font-size: 16px;
        }
        
        /* Results Grid */
        .results-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 32px;
        }
        
        /* Photo Preview Card */
        .photo-card {
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 24px;
            padding: 24px;
            box-shadow: var(--shadow-sm);
        }
        .photo-frame {
            position: relative;
            aspect-ratio: 35/45;
            border-radius: 16px;
            overflow: hidden;
            background: var(--muted);
            margin-bottom: 16px;
        }
        .photo-frame img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        .watermark {
            position: absolute;
            inset: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            background: repeating-linear-gradient(
                45deg,
                transparent,
                transparent 20px,
                hsla(24, 95%, 53%, 0.05) 20px,
                hsla(24, 95%, 53%, 0.05) 40px
            );
        }
        .watermark span {
            color: hsla(24, 95%, 53%, 0.5);
            font-size: 20px;
            font-weight: 800;
            transform: rotate(-30deg);
            text-transform: uppercase;
            letter-spacing: 4px;
        }
        .photo-specs {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 16px;
            font-size: 14px;
            color: var(--muted-foreground);
        }
        .photo-specs .dot {
            width: 4px;
            height: 4px;
            border-radius: 50%;
            background: var(--muted-foreground);
        }
        
        /* Sidebar */
        .sidebar {
            display: flex;
            flex-direction: column;
            gap: 24px;
        }
        
        /* Verification Card */
        .verify-card {
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 24px;
            padding: 24px;
            box-shadow: var(--shadow-sm);
        }
        .verify-title {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 20px;
        }
        .verify-title svg { width: 20px; height: 20px; color: var(--primary); }
        .verify-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 16px;
        }
        .verify-item {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
        }
        .verify-item svg { width: 18px; height: 18px; color: hsl(142, 76%, 36%); }
        
        /* Buttons */
        .btn {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            font-size: 16px;
            font-weight: 600;
            padding: 16px 24px;
            border-radius: 16px;
            border: none;
            cursor: pointer;
            transition: all 0.2s;
            text-decoration: none;
            width: 100%;
        }
        .btn-primary {
            background: var(--gradient-primary);
            color: white;
            box-shadow: var(--shadow-glow);
        }
        .btn-primary:hover { transform: translateY(-2px); }
        .btn-outline {
            background: white;
            border: 2px solid var(--border);
            color: var(--foreground);
        }
        .btn-outline:hover { background: var(--muted); }
        .btn svg { width: 20px; height: 20px; }
        
        .action-buttons {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }
        
        /* Trust Row */
        .trust-row {
            display: flex;
            justify-content: center;
            gap: 32px;
            padding-top: 20px;
            border-top: 1px solid var(--border);
        }
        .trust-item {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
            color: var(--muted-foreground);
        }
        .trust-item svg { width: 18px; height: 18px; }
        .trust-item svg.green { color: hsl(142, 76%, 36%); }
        .trust-item svg.orange { color: var(--primary); }
        
        /* Email Form */
        .email-form {
            background: var(--muted);
            border-radius: 16px;
            padding: 24px;
            display: none;
        }
        .email-form.active { display: block; }
        .email-form label {
            display: block;
            font-size: 14px;
            font-weight: 600;
            margin-bottom: 8px;
        }
        .email-form input {
            width: 100%;
            padding: 14px 16px;
            font-size: 16px;
            border: 2px solid var(--border);
            border-radius: 12px;
            background: white;
            color: var(--foreground);
            margin-bottom: 16px;
            transition: border-color 0.2s;
        }
        .email-form input:focus {
            outline: none;
            border-color: var(--primary);
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .results-grid {
                grid-template-columns: 1fr;
            }
            .success-header h1 { font-size: 24px; }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <header class="header">
        <div class="header-inner">
            <a href="index.php" class="back-btn">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M19 12H5M12 19l-7-7 7-7"/></svg>
                Back
            </a>
            <a href="index.php" class="logo">photo<span class="gradient-text">Aid</span></a>
            <div style="width:48px"></div>
        </div>
    </header>

    <!-- Main Content -->
    <main class="main">
        <!-- Success Header -->
        <div class="success-header">
            <div class="success-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
            </div>
            <h1>Your photo is ready!</h1>
            <p>AI-verified and HMPO compliant</p>
        </div>

        <!-- Results Grid -->
        <div class="results-grid">
            <!-- Photo Preview -->
            <div class="photo-card">
                <div class="photo-frame">
                    <img id="previewImage" src="" alt="Your passport photo">
                    <div class="watermark">
                        <span>Preview</span>
                    </div>
                </div>
                <div class="photo-specs">
                    <span>35mm × 45mm</span>
                    <span class="dot"></span>
                    <span>600 DPI</span>
                    <span class="dot"></span>
                    <span>JPEG</span>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="sidebar">
                <!-- Verification Results -->
                <div class="verify-card">
                    <div class="verify-title" id="verifyTitle">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/></svg>
                        <span>AI Verification Results</span>
                        <span id="verifyScore" style="margin-left:auto;font-size:14px;color:var(--muted-foreground);"></span>
                    </div>
                    <div class="verify-grid" id="verifyChecks">
                        <!-- Filled by JS -->
                    </div>
                    <div id="verifySuggestions" style="margin-top:16px;display:none;">
                        <p style="font-size:12px;color:var(--muted-foreground);margin-bottom:8px;">Suggestions:</p>
                        <ul id="suggestionsList" style="font-size:13px;color:#ea580c;padding-left:20px;"></ul>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="action-buttons">
                    <button class="btn btn-primary" onclick="selectPlan('digital')">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/></svg>
                        Get Digital Only — £2.99
                    </button>
                    <button class="btn btn-outline" onclick="selectPlan('prints')">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="6 9 6 2 18 2 18 9"/><path d="M6 18H4a2 2 0 0 1-2-2v-5a2 2 0 0 1 2-2h16a2 2 0 0 1 2 2v5a2 2 0 0 1-2 2h-2"/><rect x="6" y="14" width="12" height="8"/></svg>
                        Digital + 4 Prints — £5.99
                    </button>
                </div>

                <!-- Trust Badges -->
                <div class="trust-row">
                    <div class="trust-item">
                        <svg class="green" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/></svg>
                        100% Guarantee
                    </div>
                    <div class="trust-item">
                        <svg class="orange" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/></svg>
                        Instant delivery
                    </div>
                </div>

                <!-- Email Form -->
                <div class="email-form" id="emailForm">
                    <label>Enter your email to receive your photos:</label>
                    <input type="email" id="emailInput" placeholder="your@email.com">
                    <button class="btn btn-primary" onclick="sendPhotos()">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 2L11 13"/><path d="M22 2l-7 20-4-9-9-4 20-7z"/></svg>
                        Send My Photos
                    </button>
                </div>
            </div>
        </div>
    </main>

    <script>
        // Get session data from URL params
        const params = new URLSearchParams(window.location.search);
        const sessionId = params.get('session');
        const previewUrl = params.get('preview');
        
        // Set preview image
        if (previewUrl) {
            document.getElementById('previewImage').src = previewUrl;
        }
        
        // Load and display compliance data
        loadComplianceData();
        
        async function loadComplianceData() {
            try {
                // Fetch session data to get compliance info
                const response = await fetch(`get_session.php?session=${sessionId}`);
                const data = await response.json();
                
                if (data.compliance) {
                    displayCompliance(data.compliance);
                } else {
                    displayDefaultChecks();
                }
            } catch (e) {
                displayDefaultChecks();
            }
        }
        
        function displayCompliance(compliance) {
            const checksContainer = document.getElementById('verifyChecks');
            const scoreEl = document.getElementById('verifyScore');
            const titleEl = document.getElementById('verifyTitle');
            
            // Show score
            if (compliance.score) {
                scoreEl.textContent = compliance.score + '/100';
            }
            
            // Update title based on overall result
            if (compliance.overall === 'FAIL') {
                titleEl.querySelector('svg').style.color = '#ea580c';
            }
            
            // Display checks
            if (compliance.checks && compliance.checks.length > 0) {
                checksContainer.innerHTML = compliance.checks.map(check => {
                    const isPassing = check.status === 'PASS' || check.status === 'N/A';
                    const icon = isPassing 
                        ? '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="color:hsl(142,76%,36%)"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>'
                        : '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="color:#ea580c"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>';
                    
                    return `<div class="verify-item" title="${check.note || ''}">${icon} ${check.requirement}</div>`;
                }).join('');
            } else {
                displayDefaultChecks();
            }
            
            // Show suggestions if any
            if (compliance.suggestions && compliance.suggestions.length > 0) {
                const suggestionsEl = document.getElementById('verifySuggestions');
                const listEl = document.getElementById('suggestionsList');
                suggestionsEl.style.display = 'block';
                listEl.innerHTML = compliance.suggestions.map(s => `<li>${s}</li>`).join('');
            }
        }
        
        function displayDefaultChecks() {
            const checksContainer = document.getElementById('verifyChecks');
            const defaultChecks = ['Face position', 'Background', 'Lighting', 'Expression', 'Head size', 'HMPO compliant'];
            checksContainer.innerHTML = defaultChecks.map(check => 
                `<div class="verify-item">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="color:hsl(142,76%,36%)"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
                    ${check}
                </div>`
            ).join('');
        }
        
        let selectedPlan = 'digital';
        
        function selectPlan(plan) {
            selectedPlan = plan;
            document.getElementById('emailForm').classList.add('active');
            document.getElementById('emailInput').focus();
        }
        
        async function sendPhotos() {
            const email = document.getElementById('emailInput').value;
            if (!email || !email.includes('@')) {
                alert('Please enter a valid email address');
                return;
            }
            
            const btn = document.querySelector('#emailForm button');
            const originalText = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = 'Sending...';
            
            try {
                const formData = new FormData();
                formData.append('session_id', sessionId);
                formData.append('email', email);
                formData.append('plan', selectedPlan);
                
                const response = await fetch('send_photos.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    window.location.href = 'success.php?email=' + encodeURIComponent(email);
                } else {
                    alert(result.error || 'Failed to send. Please try again.');
                    btn.disabled = false;
                    btn.innerHTML = originalText;
                }
            } catch (e) {
                alert('Connection error. Please try again.');
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        }
    </script>
</body>
</html>
